<?php
/**
 * System Information Page.
 *
 * Displays detailed server information for troubleshooting.
 *
 * @package WP_WebP_Optimizer
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Check user capabilities
if ( ! current_user_can( 'manage_options' ) ) {
	wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'Mak8it.com' ) );
}

/**
 * Get detailed WebP support information
 */
function wp_webp_get_detailed_info() {
	$info = array();
	
	// PHP Version
	$info['php_version'] = PHP_VERSION;
	
	// GD Library Info
	$info['gd_loaded'] = extension_loaded( 'gd' );
	if ( $info['gd_loaded'] && function_exists( 'gd_info' ) ) {
		$gd_info = gd_info();
		$info['gd_version'] = isset( $gd_info['GD Version'] ) ? $gd_info['GD Version'] : 'Unknown';
		$info['gd_webp'] = isset( $gd_info['WebP Support'] ) ? $gd_info['WebP Support'] : false;
		$info['gd_jpeg'] = isset( $gd_info['JPEG Support'] ) ? $gd_info['JPEG Support'] : false;
		$info['gd_png'] = isset( $gd_info['PNG Support'] ) ? $gd_info['PNG Support'] : false;
	}
	
	// Function checks
	$info['imagewebp_exists'] = function_exists( 'imagewebp' );
	$info['imagecreatefromjpeg_exists'] = function_exists( 'imagecreatefromjpeg' );
	$info['imagecreatefrompng_exists'] = function_exists( 'imagecreatefrompng' );
	
	// Imagick Info
	$info['imagick_loaded'] = extension_loaded( 'imagick' );
	$info['imagick_class'] = class_exists( 'Imagick' );
	if ( $info['imagick_loaded'] && $info['imagick_class'] ) {
		$imagick = new Imagick();
		$formats = $imagick->queryFormats();
		$info['imagick_formats'] = $formats;
		$info['imagick_webp'] = in_array( 'WEBP', $formats, true );
		$imagick->clear();
	}
	
	return $info;
}

$detailed_info = wp_webp_get_detailed_info();
$support = WP_WebP_Optimizer_Converter::check_webp_support();

// Check overall status
$has_webp = false;
if ( $detailed_info['gd_loaded'] && $detailed_info['gd_webp'] && $detailed_info['imagewebp_exists'] ) {
	$has_webp = true;
} elseif ( $detailed_info['imagick_loaded'] && $detailed_info['imagick_class'] && $detailed_info['imagick_webp'] ) {
	$has_webp = true;
}
?>

<div class="wrap webp-bulk-converter-wrap">
	<h1><?php esc_html_e( 'System Information', 'small-image' ); ?></h1>

	<div class="webp-bulk-content">
		<!-- Statistics Overview -->
		<div class="webp-stats-cards">
			<div class="stat-card <?php echo $has_webp ? 'success' : 'error'; ?>">
				<div class="stat-icon">
					<span class="dashicons dashicons-<?php echo $has_webp ? 'yes-alt' : 'warning'; ?>"></span>
				</div>
				<div class="stat-details">
					<div class="stat-value"><?php echo $has_webp ? esc_html__( 'Supported', 'Mak8it.com' ) : esc_html__( 'Not Supported', 'Mak8it.com' ); ?></div>
					<div class="stat-label"><?php esc_html_e( 'WebP Conversion', 'Mak8it.com' ); ?></div>
				</div>
			</div>

			<div class="stat-card <?php echo $detailed_info['gd_loaded'] ? 'success' : 'error'; ?>">
				<div class="stat-icon">
					<span class="dashicons dashicons-<?php echo $detailed_info['gd_loaded'] ? 'yes-alt' : 'warning'; ?>"></span>
				</div>
				<div class="stat-details">
					<div class="stat-value"><?php echo $detailed_info['gd_loaded'] ? esc_html__( 'Loaded', 'Mak8it.com' ) : esc_html__( 'Not Loaded', 'Mak8it.com' ); ?></div>
					<div class="stat-label"><?php esc_html_e( 'GD Library', 'Mak8it.com' ); ?></div>
				</div>
			</div>

			<div class="stat-card <?php echo $detailed_info['imagick_loaded'] ? 'success' : 'warning'; ?>">
				<div class="stat-icon">
					<span class="dashicons dashicons-<?php echo $detailed_info['imagick_loaded'] ? 'yes-alt' : 'clock'; ?>"></span>
				</div>
				<div class="stat-details">
					<div class="stat-value"><?php echo $detailed_info['imagick_loaded'] ? esc_html__( 'Loaded', 'Mak8it.com' ) : esc_html__( 'Not Loaded', 'Mak8it.com' ); ?></div>
					<div class="stat-label"><?php esc_html_e( 'Imagick', 'Mak8it.com' ); ?></div>
				</div>
			</div>

			<div class="stat-card info">
				<div class="stat-icon">
					<span class="dashicons dashicons-info"></span>
				</div>
				<div class="stat-details">
					<div class="stat-value"><?php echo esc_html( $detailed_info['php_version'] ); ?></div>
					<div class="stat-label"><?php esc_html_e( 'PHP Version', 'Mak8it.com' ); ?></div>
				</div>
			</div>
		</div>

		<!-- Main Content -->
		<div class="webp-converter-main">
			<div class="webp-converter-card">
				<h2><?php esc_html_e( 'Server Configuration', 'Mak8it.com' ); ?></h2>

				<!-- GD Library Details -->
				<div class="info-section">
					<h3><?php esc_html_e( 'GD Library', 'Mak8it.com' ); ?></h3>
					<table class="info-table">
						<tbody>
							<tr>
								<td><strong><?php esc_html_e( 'GD Extension Loaded', 'Mak8it.com' ); ?></strong></td>
								<td>
									<?php if ( $detailed_info['gd_loaded'] ) : ?>
										<span class="status-success">✓ <?php esc_html_e( 'Yes', 'Mak8it.com' ); ?></span>
									<?php else : ?>
										<span class="status-error">✗ <?php esc_html_e( 'No', 'Mak8it.com' ); ?></span>
									<?php endif; ?>
								</td>
							</tr>
							<?php if ( $detailed_info['gd_loaded'] ) : ?>
							<tr>
								<td><strong><?php esc_html_e( 'GD Version', 'Mak8it.com' ); ?></strong></td>
								<td><?php echo esc_html( $detailed_info['gd_version'] ); ?></td>
							</tr>
							<tr>
								<td><strong><?php esc_html_e( 'WebP Support', 'Mak8it.com' ); ?></strong></td>
								<td>
									<?php if ( $detailed_info['gd_webp'] ) : ?>
										<span class="status-success">✓ <?php esc_html_e( 'Enabled', 'Mak8it.com' ); ?></span>
									<?php else : ?>
										<span class="status-error">✗ <?php esc_html_e( 'Disabled', 'Mak8it.com' ); ?></span>
									<?php endif; ?>
								</td>
							</tr>
							<tr>
								<td><strong><?php esc_html_e( 'JPEG Support', 'Mak8it.com' ); ?></strong></td>
								<td>
									<?php if ( $detailed_info['gd_jpeg'] ) : ?>
										<span class="status-success">✓ <?php esc_html_e( 'Enabled', 'Mak8it.com' ); ?></span>
									<?php else : ?>
										<span class="status-error">✗ <?php esc_html_e( 'Disabled', 'Mak8it.com' ); ?></span>
									<?php endif; ?>
								</td>
							</tr>
							<tr>
								<td><strong><?php esc_html_e( 'PNG Support', 'Mak8it.com' ); ?></strong></td>
								<td>
									<?php if ( $detailed_info['gd_png'] ) : ?>
										<span class="status-success">✓ <?php esc_html_e( 'Enabled', 'Mak8it.com' ); ?></span>
									<?php else : ?>
										<span class="status-error">✗ <?php esc_html_e( 'Disabled', 'Mak8it.com' ); ?></span>
									<?php endif; ?>
								</td>
							</tr>
							<tr>
								<td><strong><?php esc_html_e( 'imagewebp() function', 'Mak8it.com' ); ?></strong></td>
								<td>
									<?php if ( $detailed_info['imagewebp_exists'] ) : ?>
										<span class="status-success">✓ <?php esc_html_e( 'Available', 'Mak8it.com' ); ?></span>
									<?php else : ?>
										<span class="status-error">✗ <?php esc_html_e( 'Not Available', 'Mak8it.com' ); ?></span>
									<?php endif; ?>
								</td>
							</tr>
							<?php endif; ?>
						</tbody>
					</table>
				</div>

				<!-- Imagick Details -->
				<div class="info-section">
					<h3><?php esc_html_e( 'Imagick', 'Mak8it.com' ); ?></h3>
					<table class="info-table">
						<tbody>
							<tr>
								<td><strong><?php esc_html_e( 'Imagick Extension Loaded', 'Mak8it.com' ); ?></strong></td>
								<td>
									<?php if ( $detailed_info['imagick_loaded'] ) : ?>
										<span class="status-success">✓ <?php esc_html_e( 'Yes', 'Mak8it.com' ); ?></span>
									<?php else : ?>
										<span class="status-error">✗ <?php esc_html_e( 'No', 'Mak8it.com' ); ?></span>
									<?php endif; ?>
								</td>
							</tr>
							<tr>
								<td><strong><?php esc_html_e( 'Imagick Class Available', 'Mak8it.com' ); ?></strong></td>
								<td>
									<?php if ( $detailed_info['imagick_class'] ) : ?>
										<span class="status-success">✓ <?php esc_html_e( 'Yes', 'Mak8it.com' ); ?></span>
									<?php else : ?>
										<span class="status-error">✗ <?php esc_html_e( 'No', 'Mak8it.com' ); ?></span>
									<?php endif; ?>
								</td>
							</tr>
							<?php if ( $detailed_info['imagick_loaded'] && $detailed_info['imagick_class'] ) : ?>
							<tr>
								<td><strong><?php esc_html_e( 'WebP Format Support', 'Mak8it.com' ); ?></strong></td>
								<td>
									<?php if ( $detailed_info['imagick_webp'] ) : ?>
										<span class="status-success">✓ <?php esc_html_e( 'Enabled', 'Mak8it.com' ); ?></span>
									<?php else : ?>
										<span class="status-error">✗ <?php esc_html_e( 'Disabled', 'Mak8it.com' ); ?></span>
									<?php endif; ?>
								</td>
							</tr>
							<?php endif; ?>
						</tbody>
					</table>
				</div>

				<!-- Recommendations -->
				<?php if ( ! $has_webp ) : ?>
				<div class="recommendations-section">
					<h3><?php esc_html_e( 'How to Enable WebP Support', 'Mak8it.com' ); ?></h3>
					<div class="conversion-info" style="background: #fcf0f1; border-left-color: #d63638;">
						<p class="info-message" style="color: #b32d2e;">
							<span class="dashicons dashicons-warning"></span>
							<strong><?php esc_html_e( 'WebP conversion is NOT currently supported on your server.', 'Mak8it.com' ); ?></strong>
						</p>
					</div>

					<p><strong><?php esc_html_e( 'To enable WebP support, you need ONE of the following:', 'Mak8it.com' ); ?></strong></p>
					<ol class="recommendations-list">
						<?php if ( ! $detailed_info['gd_loaded'] ) : ?>
							<li><?php esc_html_e( 'Install PHP GD extension', 'Mak8it.com' ); ?></li>
						<?php elseif ( ! $detailed_info['gd_webp'] ) : ?>
							<li><?php esc_html_e( 'Recompile PHP GD with WebP support (--with-webp)', 'Mak8it.com' ); ?></li>
						<?php endif; ?>
						<?php if ( ! $detailed_info['imagick_loaded'] ) : ?>
							<li><?php esc_html_e( 'Install PHP Imagick extension', 'Mak8it.com' ); ?></li>
						<?php elseif ( ! $detailed_info['imagick_webp'] ) : ?>
							<li><?php esc_html_e( 'Install ImageMagick with WebP delegate support', 'Mak8it.com' ); ?></li>
						<?php endif; ?>
					</ol>

					<div class="xampp-instructions">
						<h4><?php esc_html_e( 'For XAMPP Users (Windows):', 'Mak8it.com' ); ?></h4>
						<ol>
							<li><?php esc_html_e( 'Download libwebp DLL files from https://developers.google.com/speed/webp/download', 'Mak8it.com' ); ?></li>
							<li><?php esc_html_e( 'Copy libwebp.dll and libwebp_x64.dll to C:\\xampp\\php\\ext\\', 'Mak8it.com' ); ?></li>
							<li><?php esc_html_e( 'Edit C:\\xampp\\php\\php.ini and ensure extension=gd is enabled (uncommented)', 'Mak8it.com' ); ?></li>
							<li><?php esc_html_e( 'Restart Apache from XAMPP Control Panel', 'Mak8it.com' ); ?></li>
							<li><?php esc_html_e( 'Refresh this page to check if WebP is now supported', 'Mak8it.com' ); ?></li>
						</ol>
						<p class="note"><em><?php esc_html_e( 'Note: In newer PHP versions (7.4+), GD should include WebP support by default if libwebp is available.', 'Mak8it.com' ); ?></em></p>
					</div>
				</div>
				<?php else : ?>
				<div class="conversion-info" style="background: #e5f5ea; border-left-color: #00a32a;">
					<p class="info-message" style="color: #007017;">
						<span class="dashicons dashicons-yes-alt"></span>
						<strong><?php echo esc_html( $support['message'] ); ?></strong>
						<strong>(<?php echo esc_html( $support['method'] ); ?>)</strong>
					</p>
				</div>
				<?php endif; ?>
			</div>

			<!-- Sidebar -->
			<div class="webp-converter-sidebar">
				<div class="sidebar-card">
					<h3><?php esc_html_e( 'System Details', 'Mak8it.com' ); ?></h3>
					<ul class="quick-stats-list">
						<li>
							<strong><?php esc_html_e( 'PHP Version', 'Mak8it.com' ); ?></strong>
							<span><?php echo esc_html( PHP_VERSION ); ?></span>
						</li>
						<li>
							<strong><?php esc_html_e( 'Memory Limit', 'Mak8it.com' ); ?></strong>
							<span><?php echo esc_html( ini_get( 'memory_limit' ) ); ?></span>
						</li>
						<li>
							<strong><?php esc_html_e( 'Max Upload Size', 'Mak8it.com' ); ?></strong>
							<span><?php echo esc_html( size_format( wp_max_upload_size() ) ); ?></span>
						</li>
						<li>
							<strong><?php esc_html_e( 'WordPress Version', 'Mak8it.com' ); ?></strong>
							<span><?php echo esc_html( get_bloginfo( 'version' ) ); ?></span>
						</li>
					</ul>
				</div>

				<div class="sidebar-card info-card">
					<h3><?php esc_html_e( 'Quick Actions', 'Mak8it.com' ); ?></h3>
					<p style="margin-bottom: 10px;">
						<a href="<?php echo esc_url( admin_url( 'admin.php?page=small-image' ) ); ?>" class="button button-secondary" style="width: 100%; text-align: center;">
							<?php esc_html_e( 'Settings', 'Mak8it.com' ); ?>
						</a>
					</p>
					<p style="margin-bottom: 10px;">
						<a href="<?php echo esc_url( admin_url( 'admin.php?page=small-image' ) ); ?>" class="button button-secondary" style="width: 100%; text-align: center;">
							<?php esc_html_e( 'Dashboard', 'small-image' ); ?>
						</a>
					</p>
					<p>
						<a href="<?php echo esc_url( admin_url( 'admin.php?page=small-image-system-info' ) ); ?>" class="button button-secondary" style="width: 100%; text-align: center;" onclick="location.reload(); return false;">
							<?php esc_html_e( 'Refresh Info', 'Mak8it.com' ); ?>
						</a>
					</p>
				</div>

				<div class="sidebar-card">
					<h3><?php esc_html_e( 'Support Status', 'Mak8it.com' ); ?></h3>
					<?php if ( $has_webp ) : ?>
						<p class="status-success">
							<span class="dashicons dashicons-yes-alt"></span>
							<?php esc_html_e( 'Your server is ready for WebP conversion!', 'Mak8it.com' ); ?>
						</p>
					<?php else : ?>
						<p class="status-error">
							<span class="dashicons dashicons-warning"></span>
							<?php esc_html_e( 'WebP support needs to be enabled. Follow the instructions on this page.', 'Mak8it.com' ); ?>
						</p>
					<?php endif; ?>
				</div>
			</div>
		</div>
	</div>
</div>
